<?php

namespace app\models;

use Yii;
use yii\web\UploadedFile;

/**
 * This is the model class for table "travelerprofile".
 *
 * @property int $id
 * @property int $age
 * @property string $travel_preferences
 * @property int $interests_id
 * @property int $budget_id
 * @property string $bio
 * @property string $contact_info
 * @property int $user_id
 * @property string|null $image Путь к изображению
 *
 * @property Budget $budget
 * @property Interests $interests
 * @property User $user
 */
class Travelerprofile extends \yii\db\ActiveRecord
{
    /**
     * @var UploadedFile Атрибут для загрузки файла
     */
    public $imageFile;

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'travelerprofile';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['age', 'interests_id', 'budget_id', 'user_id'], 'integer'],
            [['travel_preferences'], 'string', 'max' => 800],
            [['bio'], 'string', 'max' => 1000],
            [['contact_info', 'image'], 'string', 'max' => 255],
            [['budget_id'], 'exist', 'skipOnError' => true, 'targetClass' => Budget::class, 'targetAttribute' => ['budget_id' => 'id']],
            [['interests_id'], 'exist', 'skipOnError' => true, 'targetClass' => Interests::class, 'targetAttribute' => ['interests_id' => 'id']],
            ['user_id', 'default', 'value' => Yii::$app->user->getId()],
            [['imageFile'], 'file', 'skipOnEmpty' => true, 'extensions' => 'png, jpg, jpeg', 'maxSize' => 1024 * 1024 * 2], // 2MB
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'age' => 'Возраст',
            'travel_preferences' => 'Предпочтения в путешествиях',
            'interests_id' => 'Интересы',
            'budget_id' => 'Бюджет',
            'bio' => 'О себе',
            'contact_info' => 'Контактная информация',
            'user_id' => 'User ID',
            'image' => 'Фото',
            'imageFile' => 'Upload Image',
        ];
    }

    /**
     * Обработка загрузки изображения
     */
    public function upload()
    {
        if ($this->imageFile) {
            $path = Yii::getAlias('@webroot/image/');
            if (!file_exists($path)) {
                mkdir($path, 0777, true);
            }

            $filename = Yii::$app->security->generateRandomString() . '.' . $this->imageFile->extension;
            $this->imageFile->saveAs($path . $filename);
            $this->image = '/image/' . $filename;
            return true;
        }
        return false;
    }

    /**
     * {@inheritdoc}
     */
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            $this->imageFile = UploadedFile::getInstance($this, 'imageFile');
            if ($this->imageFile) {
                // Удаляем старое изображение, если оно есть
                if (!$this->isNewRecord && $this->getOldAttribute('image')) {
                    $oldImage = Yii::getAlias('@webroot') . $this->getOldAttribute('image');
                    if (file_exists($oldImage)) {
                        unlink($oldImage);
                    }
                }
                $this->upload();
            }
            return true;
        }
        return false;
    }

    /**
     * {@inheritdoc}
     */
    public function afterDelete()
    {
        parent::afterDelete();
        // Удаляем изображение при удалении профиля
        if ($this->image) {
            $imagePath = Yii::getAlias('@webroot') . $this->image;
            if (file_exists($imagePath)) {
                unlink($imagePath);
            }
        }
    }

    /**
     * Gets query for [[Budget]].
     *
     * @return \yii\db\ActiveQuery
     */
    public function getBudget()
    {
        return $this->hasOne(Budget::class, ['id' => 'budget_id']);
    }

    /**
     * Gets query for [[Interests]].
     *
     * @return \yii\db\ActiveQuery
     */
    public function getInterests()
    {
        return $this->hasOne(Interests::class, ['id' => 'interests_id']);
    }

    /**
     * Gets query for [[User]].
     *
     * @return \yii\db\ActiveQuery
     */
    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    /**
     * Получает URL изображения профиля
     */
    public function getImageUrl()
    {
        return $this->image ? Yii::getAlias('@web') . $this->image : '/default-avatar.jpg';
    }
}